#ifndef OPENGL_LIB_H_INCLUDED
#define OPENGL_LIB_H_INCLUDED

#include <vector>

#define WHEELSPEED 0.5

#define LEFTDRAGSPEED 0.5
#define RIGHTDRAGSPEED 0.25
#define ROTATESPEED 0.5

#define FOVY 45.0
#define NEAR_CLIPPING_DISTANCE 0.1
#define FAR_CLIPPING_DISTANCE 1000

namespace genv {

    //enum for draw modes
    typedef enum DrawModeType
    {
        DRAW_POINTS,
        DRAW_LINES,
        DRAW_TRIANGLES,
        DRAW_WIREFRAME
    } DrawMode;

    //PART OF "gout", CHANGED!!!
    // Event descriptor
    struct event
    {
        int keycode;
        int pos_x, pos_y;
        int button;
        int time;
        int type;

        //CHANGED
        int rel_x,rel_y;
    };


    class GLcanvas
    {
    public:

        //returns an instance
        static GLcanvas& instance();

        //open opengl window
        void open(int Width,int Height);

        //assign vertices, colors and indices to the library, set drawmode.
        //Each vertex has 3 position values, 3 color values. Number of indices is arbitrary
        void setVertices(std::vector<float> Vertices,std::vector<float> Colors, std::vector<unsigned int> Indices, DrawMode Mode);

        //implemented mouse event handler. Use left, middle, right buttons and the wheel to rotate or move object
        //Hold left button: rotate the object around the horizontal and vertical axis
        //hold right button: rotate the object around the z-axis (perpendicular to screen)
        //hold middle button: move the object horizontally and vertically
        //wheel: move the object closer or farther
        //hold left and right button simultaneously: rotate object around the viewpoint
        void mouseEventHandler(const event &ev);

        //refresh the screen (redraw, repaint)
        void refresh();

        void show();

        //ask for a timer. The wait time is milliseconds
        void timer(int wait);

        //switch lighting on and off
        void lighting(bool on);

        //operator void* -> returns 0 if window is closed
        operator const void*() const
        { if (quit) return 0; else return this; }


        //rotate object around the x,y and z axis, where the axes are
        //going through the object center
        //
        //x is horizontal (right positive), y is vertical (up positive),
        //z is perpendicular to screen (out of screen positive)
        void rotate_abs_OC(float x,float y,float z);


        //rotate object around the x,y and z axis, where the axes are
        //going through the viewpoint
        //
        //x is horizontal (right positive), y is vertical (up positive),
        //z is perpendicular to screen (out of screen positive)
        void rotate_abs_Viewer(float x,float y,float z);


        //rotate object around the x and y axis, where the axes are
        //going through the viewpoint.
        //
        //x is horizontal (right positive), y is vertical (up positive)
        //
        //The unit of the rotation is pixels
        //on the screen, so this is useful when we'd like the object center
        //to follow the mouse cursor
        void rotate_abs_Viewer_pixel(float x,float y);


        //rotate object around the OBJECT's x,y and z axis, where the axes are
        //going through the object center.
        //
        //At startup, the object's axes are the same as in the earlier functions,
        //but they move as the object moves
        void rotate_rel_OC(float x,float y,float z);

        //Move the object in its own x,y and z directions.
        //
        //At startup, the object's axes are the same as in the earlier functions,
        //but they move as the object moves
        void translate_rel(float x,float y,float z);


        //Move the object in x,y and z directions.
        //
        //x is horizontal (right positive), y is vertical (up positive),
        //z is perpendicular to screen (out of screen positive)
        void translate_abs(float x,float y,float z);


        //Move the object in x and y directions.
        //
        //x is horizontal (right positive), y is vertical (up positive)
        //
        //The unit of the translation is pixels
        //on the screen, so this is useful when we'd like the object center
        //to follow the mouse cursor
        void translate_abs_pixel(float x,float y);

        //wait for event (called by operator>>)
        GLcanvas& wait_event(genv::event&);

    private:

        //after calling setup_opengl, these will store
        //the field of view (in degrees!!!) in x and y directions
        float fovx,fovy;

        //Distance per pixel: multiplied by the Z distance of
        //the object, this stores the amount of movement needed
        //so that the abject center would move 1 pixel
        //
        //stored for computational reasons only
        float DistancePerPixel;

        //constructor
        GLcanvas();

        //sets up sdl window (does not open it)
        void setup_sdl();

        //sets up opengl environment
        void setup_opengl();

        //stores the inverse of the projection matrix
        //(openGL does not provide this by default)
        float InvProjMat[16];

        //mouse button indicators
        bool LeftButtonHeldDown;
        bool MiddleButtonHeldDown;
        bool RightButtonHeldDown;

        bool Lighting;

        //width and height of opengl window
        int Width,Height;

        //stores assigned Colors, Vertices and Indices
        std::vector<float> Colors;
        std::vector<float> Vertices;
        std::vector<unsigned int> Indices;
        std::vector<float> Normals;

        //stores the draw mode
        DrawMode Mode;

        //indicating window closure
        bool quit;
    };

    extern GLcanvas& glout;

    //PART OF "gout"
    inline genv::GLcanvas& operator >> (genv::GLcanvas& inp, genv::event& ev)
    {
        return inp.wait_event(ev);
    }

    //PART OF "gout"
    enum keycode_t {
        key_tab = '\t', key_backspace = '\b', key_enter = '\r',
        key_escape = '\033', key_space = ' ',
        key_up = 0x10000, key_down, key_right, key_left,
        key_insert, key_delete, key_home, key_end, key_pgup, key_pgdn,
        key_lshift, key_rshift, key_lctrl, key_rctrl, key_lalt, key_ralt,
        key_lwin, key_rwin, key_menu, key_numl, key_capsl, key_scrl,
        key_f0 = 0x20000, key_f1, key_f2, key_f3, key_f4, key_f5, key_f6, key_f7,
        key_f8, key_f9, key_f10, key_f11, key_f12, key_f13, key_f14, key_f15
    };

    //PART OF "gout"
    enum button_t {
        btn_left = 1, btn_middle, btn_right, btn_wheelup, btn_wheeldown
    };

    //PART OF "gout"
    enum event_type {
        ev_key = 1, ev_mouse, ev_timer
    };
}

#endif // OPENGL_LIB_H_INCLUDED
