#include "OpenGL_lib.h"

#include "SDL/SDL.h"
#include "SDL/SDL_opengl.h"

#include <iostream>
#include <cmath>

genv::GLcanvas& genv::glout = genv::GLcanvas::instance();

namespace {
    SDL_TimerID timer_id;

    //PART OF "gout"
     Uint32 timer_event(Uint32 interval, void*)
    {
        SDL_Event ev;
        ev.type = SDL_USEREVENT;
        ev.user.code = SDL_GetTicks();
        ev.user.data1 = 0;
        ev.user.data2 = 0;
        SDL_PushEvent(&ev);
        return interval;
    }
}

//PART OF "gout"
void genv::GLcanvas::timer(int wait)
{
    if (timer_id != 0)
    {
        SDL_RemoveTimer(timer_id);
        timer_id = 0;
    }

    if (wait > 0)
    {
        timer_id = SDL_AddTimer(wait, timer_event, 0);
        if (timer_id==NULL) {
            std::cout << "err:" <<std::endl << SDL_GetError() <<std::endl <<std::endl;
        }
    }
}

//PART OF "gout", CHANGED
genv::GLcanvas& genv::GLcanvas::wait_event(event& ev)
{
    static event nullev = { 0, 0, 0, 0, 0, 0, 0 };
    ev = nullev;
    if (quit)
        return *this;

    bool got = false;
    do
    {
        SDL_Event se;
        if (!SDL_WaitEvent(&se))
        {
            quit = true;
            return *this;
        }

        switch (se.type)
        {
            case SDL_QUIT:
                quit = true;
                got = true;
                break;
            case SDL_KEYUP:
            case SDL_KEYDOWN:
                ev.type = ev_key;
                //CHANGED -> when merging, it can be set back to the original
                //ev.keycode = mkkeycode(se.key.keysym.sym, se.key.keysym.unicode);
                ev.keycode=se.key.keysym.sym;
                ev.keycode *= (se.type == SDL_KEYUP ? -1 : 1);
                got = ev.keycode != 0;
                break;
            case SDL_MOUSEBUTTONDOWN:
            case SDL_MOUSEBUTTONUP:
                ev.type = ev_mouse;
                ev.button = se.button.button;
                ev.button *= (se.button.state == SDL_RELEASED ? -1 : 1);
                ev.pos_x = se.button.x;
                ev.pos_y = se.button.y;
                got = true;
                break;
            case SDL_MOUSEMOTION:
                ev.type = ev_mouse;
                ev.pos_x = se.motion.x;
                ev.pos_y = se.motion.y;
                //CHANGED!!!
                ev.rel_x = se.motion.xrel;
                ev.rel_y = se.motion.yrel;
                got = true;
                break;
            case SDL_USEREVENT:
                ev.type = ev_timer;
                ev.time = se.user.code;
                got = true;
                break;
            default: ;
        }
    } while (!got);

    return *this;
}


void genv::GLcanvas::setup_sdl()
{

    if ( SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER ) < 0 ) {
        fprintf(stderr, "Couldn't initialize SDL: %s\n", SDL_GetError());
        exit(1);
    }

    /* Quit SDL properly on exit */
    atexit(SDL_Quit);

    /* Set the minimum requirements for the OpenGL window */
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 5 );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 5 );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 5 );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

    /* Note the SDL_DOUBLEBUF flag is not required to enable double
     * buffering when setting an OpenGL video mode.
     * Double buffering is enabled or disabled using the
     * SDL_GL_DOUBLEBUFFER attribute.
     */

}


void genv::GLcanvas::open(int Width,int Height)
{

    this->Width=Width;
    this->Height=Height;

    /* Get the current video information */
    const SDL_VideoInfo* video;
    video = SDL_GetVideoInfo( );
    if( video == NULL ) {
        fprintf(stderr, "Couldn't get video information: %s\n", SDL_GetError());
        exit(1);
    }

    if( SDL_SetVideoMode( Width, Height, video->vfmt->BitsPerPixel, SDL_OPENGL ) == 0 ) {
        fprintf(stderr, "Couldn't set video mode: %s\n", SDL_GetError());
        exit(1);
    }

    setup_opengl();

}


void genv::GLcanvas::mouseEventHandler(const event &ev)
{
    if (ev.type!=ev_mouse) {
        return;
    }

    if (ev.button>0) {
        switch (ev.button) {
            case btn_left:
                LeftButtonHeldDown=true;
                break;
            case btn_middle:
                MiddleButtonHeldDown=true;
                break;
            case btn_right:
                RightButtonHeldDown=true;
                break;
            case btn_wheelup:
                //Distance/=1+WHEELSPEED;
                //if (Distance==0) Distance=0.0001f;
                //if (Distance<MINDIST) Distance=MINDIST;
                //if (Distance>MAXDIST) Distance=MAXDIST;
                translate_abs(0,0,1);
                break;
            case btn_wheeldown:
                //Distance*=1+WHEELSPEED;
                //if (Distance<MINDIST) Distance=MINDIST;
                //if (Distance>MAXDIST) Distance=MAXDIST;
                translate_abs(0,0,-1);
                break;
        }
    }
    if (ev.button<0) {
        switch (ev.button) {
            case -btn_left:
                LeftButtonHeldDown=false;
                break;
            case -btn_middle:
                MiddleButtonHeldDown=false;
                break;
            case -btn_right:
                RightButtonHeldDown=false;
                break;
        }
    }
    if (ev.button==0) {
        if (LeftButtonHeldDown && RightButtonHeldDown) {
            rotate_abs_Viewer_pixel(-ev.rel_y*LEFTDRAGSPEED,-ev.rel_x*LEFTDRAGSPEED);
            return;
        }
        if (LeftButtonHeldDown) {
            rotate_abs_OC      (ev.rel_y*LEFTDRAGSPEED,ev.rel_x*LEFTDRAGSPEED,0);
            //rotate_abs_Viever(ev.rel_y*LEFTDRAGSPEED,ev.rel_x*LEFTDRAGSPEED,0);
            //rotate_rel_OC    (ev.rel_y*LEFTDRAGSPEED,ev.rel_x*LEFTDRAGSPEED,0);
        }
        if (MiddleButtonHeldDown) {
            rotate_abs_OC    (0,0,-ev.rel_x*ROTATESPEED);
            //rotate_abs_Viewer(0,0,-ev.rel_x*ROTATESPEED);
            //rotate_rel_OC    (0,0,-ev.rel_x*ROTATESPEED);
        }
        if (RightButtonHeldDown) {
            translate_abs_pixel(ev.rel_x,-ev.rel_y);
        }
        //std::cout << (int)event.button.state <<std::endl;
        //SDL_MOUSEBUTTONDOWNMASK;
    }
}








void genv::GLcanvas::setVertices(std::vector<GLfloat> InputVertices,std::vector<GLfloat> InputColors, std::vector<GLuint> InputIndices, DrawMode InputMode)
{
    Mode=InputMode;

    switch (Mode)
    {
    case DRAW_TRIANGLES:
    case DRAW_WIREFRAME:
        {
        glEnable(GL_CULL_FACE);

        if (Mode==DRAW_WIREFRAME) {
            glDisable(GL_LIGHTING);
        } else {
            if (Lighting) glEnable(GL_LIGHTING); else glDisable(GL_LIGHTING);
        }

        Normals.clear();
        Vertices.clear();
        Colors.clear();
        std::vector<GLuint>::iterator Ind=InputIndices.begin();
        //reserve InputIndices.size()/3*18
        Normals.reserve(InputIndices.size()/3*18);
        Vertices.reserve(InputIndices.size()/3*18);
        Colors.reserve(InputIndices.size()/3*18);

        float x1,y1,z1,x2,y2,z2,x3,y3,z3;
        float c1r,c1g,c1b,c2r,c2g,c2b,c3r,c3g,c3b;

        for (unsigned int i=0;i<InputIndices.size()/3;i++)
        {
            x1=InputVertices[ 3*(*Ind) ];
            y1=InputVertices[ 3*(*Ind)+1 ];
            z1=InputVertices[ 3*(*Ind)+2 ];

            c1r=InputColors[ 3*(*Ind) ];
            c1g=InputColors[ 3*(*Ind)+1 ];
            c1b=InputColors[ 3*(*Ind)+2 ];

            Ind++;

            x2=InputVertices[ 3*(*Ind) ];
            y2=InputVertices[ 3*(*Ind)+1 ];
            z2=InputVertices[ 3*(*Ind)+2 ];

            c2r=InputColors[ 3*(*Ind) ];
            c2g=InputColors[ 3*(*Ind)+1 ];
            c2b=InputColors[ 3*(*Ind)+2 ];

            Ind++;

            x3=InputVertices[ 3*(*Ind) ];
            y3=InputVertices[ 3*(*Ind)+1 ];
            z3=InputVertices[ 3*(*Ind)+2 ];

            c3r=InputColors[ 3*(*Ind) ];
            c3g=InputColors[ 3*(*Ind)+1 ];
            c3b=InputColors[ 3*(*Ind)+2 ];

            Ind++;


                Vertices.push_back(x1);
                Vertices.push_back(y1);
                Vertices.push_back(z1);
                Colors.push_back(c1r);
                Colors.push_back(c1g);
                Colors.push_back(c1b);

                Vertices.push_back(x2);
                Vertices.push_back(y2);
                Vertices.push_back(z2);
                Colors.push_back(c2r);
                Colors.push_back(c2g);
                Colors.push_back(c2b);

                Vertices.push_back(x3);
                Vertices.push_back(y3);
                Vertices.push_back(z3);
                Colors.push_back(c3r);
                Colors.push_back(c3g);
                Colors.push_back(c3b);




                Vertices.push_back(x2);
                Vertices.push_back(y2);
                Vertices.push_back(z2);
                Colors.push_back(c2r);
                Colors.push_back(c2g);
                Colors.push_back(c2b);

                Vertices.push_back(x1);
                Vertices.push_back(y1);
                Vertices.push_back(z1);
                Colors.push_back(c1r);
                Colors.push_back(c1g);
                Colors.push_back(c1b);

                Vertices.push_back(x3);
                Vertices.push_back(y3);
                Vertices.push_back(z3);
                Colors.push_back(c3r);
                Colors.push_back(c3g);
                Colors.push_back(c3b);


            float Nx=(y2-y1) * (z3-z2) - (z2-z1) * (y3-y2);
            float Ny=(z2-z1) * (x3-x2) - (x2-x1) * (z3-z2);
            float Nz=(x2-x1) * (y3-y2) - (y2-y1) * (x3-x2);

            float len = (float)(sqrt((Nx * Nx) + (Ny * Ny) + (Nz * Nz)));

            // avoid division by 0
            if (len == 0.0f)
            len = 1.0f;

            // reduce to unit size
            Nx /= len;
            Ny /= len;
            Nz /= len;

            //Nx=-Nx;
            //Ny=-Ny;
            //Nz=-Nz;

            for (int i=0;i<2;i++) {
                Normals.push_back( Nx );
                Normals.push_back( Ny );
                Normals.push_back( Nz );
                Normals.push_back( Nx );
                Normals.push_back( Ny );
                Normals.push_back( Nz );
                Normals.push_back( Nx );
                Normals.push_back( Ny );
                Normals.push_back( Nz );

                Nx=-Nx;
                Ny=-Ny;
                Nz=-Nz;

            }
        }
        break;
        }
    case DRAW_LINES:
        //lines
        glDisable(GL_LIGHTING);
        glDisable(GL_CULL_FACE);

        this->Vertices=InputVertices;
        this->Colors=InputColors;
        this->Indices=InputIndices;
    case DRAW_POINTS:
        //points
        glDisable(GL_LIGHTING);
        glDisable(GL_CULL_FACE);

        this->Vertices=InputVertices;
        this->Colors=InputColors;
        Indices.clear();
        for (unsigned int i=0;i<Vertices.size()/3;i++)
        {
            Indices.push_back(i);
        }
        break;
    }
}


genv::GLcanvas::GLcanvas() : quit(false)
{
    LeftButtonHeldDown=false;
    MiddleButtonHeldDown=false;
    RightButtonHeldDown=false;

    Colors.clear();
    Vertices.clear();
    Normals.clear();

    setup_sdl();

    timer_id=0;
}


genv::GLcanvas& genv::GLcanvas::instance()
{
    static GLcanvas single_inst;
    return single_inst;
}


void genv::GLcanvas::translate_abs_pixel(float x,float y)
{
    float M[16];
    glGetFloatv(GL_MODELVIEW_MATRIX,M);
    float ZDistance=-M[14];

    float X= x * ZDistance * DistancePerPixel / sqrt(InvProjMat[0]*InvProjMat[0]+InvProjMat[1]*InvProjMat[1]+InvProjMat[2]*InvProjMat[2]);
    float Y= y * ZDistance * DistancePerPixel / sqrt(InvProjMat[4]*InvProjMat[4]+InvProjMat[5]*InvProjMat[5]+InvProjMat[6]*InvProjMat[6]);

    translate_abs(X,Y,0);
}


void genv::GLcanvas::translate_abs(float x,float y,float z)
{
    glTranslatef(x*InvProjMat[0],x*InvProjMat[1],x*InvProjMat[2]);
    glTranslatef(y*InvProjMat[4],y*InvProjMat[5],y*InvProjMat[6]);
    glTranslatef(z*InvProjMat[8],z*InvProjMat[9],z*InvProjMat[10]);

    //keep calculating inverse
    glPushMatrix();
        glLoadIdentity();
        //multiply the current inverse FROM LEFT with the inverse of the translation
        glTranslatef(-z*InvProjMat[8],-z*InvProjMat[9],-z*InvProjMat[10]);
        glTranslatef(-y*InvProjMat[4],-y*InvProjMat[5],-y*InvProjMat[6]);
        glTranslatef(-x*InvProjMat[0],-x*InvProjMat[1],-x*InvProjMat[2]);



        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();
}


void genv::GLcanvas::translate_rel(float x,float y,float z)
{
    glTranslatef(x,y,z);

    //keep calculating inverse
    glPushMatrix();
        glLoadIdentity();
        //multiply the current inverse FROM LEFT with the inverse of the translation
        glTranslatef(-x,-y,-z);
        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();
}


void genv::GLcanvas::rotate_abs_OC(float x,float y,float z)
{
    glRotatef(x,InvProjMat[0],InvProjMat[1],InvProjMat[2]);
    glRotatef(y,InvProjMat[4],InvProjMat[5],InvProjMat[6]);
    glRotatef(z,InvProjMat[8],InvProjMat[9],InvProjMat[10]);

    //keep calculating inverse
    glPushMatrix();
        glLoadIdentity();
        //multiply the current inverse FROM LEFT with the inverse of the rotation
        glRotatef(-z,InvProjMat[8],InvProjMat[9],InvProjMat[10]);
        glRotatef(-y,InvProjMat[4],InvProjMat[5],InvProjMat[6]);
        glRotatef(-x,InvProjMat[0],InvProjMat[1],InvProjMat[2]);

        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();
}


void genv::GLcanvas::rotate_abs_Viewer_pixel(float x,float y)
{
    rotate_abs_Viewer(x/Height*fovy*2,y/Width*fovx*2,0);
}


void genv::GLcanvas::rotate_abs_Viewer(float x,float y,float z)
{
    float dx,dy,dz;
    dx=InvProjMat[12];dy=InvProjMat[13];dz=InvProjMat[14];
    glTranslatef(dx,dy,dz);
    glPushMatrix();
        glLoadIdentity();
        glTranslatef(-dx,-dy,-dz);
        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();

    rotate_abs_OC(x,y,z);

    glTranslatef(-dx,-dy,-dz);
    glPushMatrix();
        glLoadIdentity();
        glTranslatef(dx,dy,dz);
        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();
}


void genv::GLcanvas::rotate_rel_OC(float x,float y,float z)
{
    glRotatef(x,1.0,0.0,0.0);
    glRotatef(y,0.0,1.0,0.0);
    glRotatef(z,0.0,0.0,1.0);

    //keep calculating inverse
    glPushMatrix();
        glLoadIdentity();
        //multiply the current inverse FROM LEFT with the inverse of the rotation
        glRotatef(-z,0.0,0.0,1.0);
        glRotatef(-y,0.0,1.0,0.0);
        glRotatef(-x,1.0,0.0,0.0);

        glMultMatrixf(InvProjMat);
        glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);
    glPopMatrix();
}


void genv::GLcanvas::lighting(bool on)
{
    Lighting=on;
    if (Mode==DRAW_TRIANGLES) {
        if (Lighting) glEnable(GL_LIGHTING); else glDisable(GL_LIGHTING);
    }
}


void genv::GLcanvas::show()
{

    //starting translation and rotation
    glout.translate_abs(0,0,-6);
    glout.rotate_abs_OC(0,0,0);

    //draw 3d object
    glout.refresh();

    //glout.timer(100);
    bool light=true;
    glout.lighting(light);
    event ev;
    while (glout>>ev) {
        if (ev.type==ev_mouse) {
            //implemented mouse event handler ->
            //users don't have to implement this
            //AND don't have to use it
            glout.mouseEventHandler(ev);
        }
        if (ev.type==ev_key)
        {
            switch (ev.keycode)
            {
                case 'l':
                    //switch lighting
                    glout.lighting(light=!light);
                    break;
                case 'w':
                    //switch wireframe mode
                    if (Mode==DRAW_WIREFRAME)
                    {
                        Mode=DRAW_TRIANGLES;
                        //glout.setVertices(Vertices,Colors,Indices,Mode);
                        if (Lighting) glEnable(GL_LIGHTING); else glDisable(GL_LIGHTING);
                    } else if (Mode==DRAW_TRIANGLES)
                    {
                        Mode=DRAW_WIREFRAME;
                        //glout.setVertices(Vertices,Colors,Indices,Mode);
                        glDisable(GL_LIGHTING);
                    }
                    break;
            }
        }
        if (ev.type==ev_timer) {
            //automatic rotation
            //glout.rotate_abs_OC(0,1,0);
        }
        //draw 3d object
        glout.refresh();
    }
}


void genv::GLcanvas::refresh()
{
    /* Clear the color plane and the z-buffer */
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glEnableClientState(GL_COLOR_ARRAY);
    glEnableClientState(GL_VERTEX_ARRAY);

    glVertexPointer(3, GL_FLOAT, 0, &Vertices.front());
    glColorPointer(3, GL_FLOAT, 0,  &Colors.front()    );



    //glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
    //GL_NORMAL_ARRAY
    //glEnableVertexAttribArray(ATTRIB_NORMAL);

    switch (Mode) {
        case DRAW_WIREFRAME:
            glDisableClientState(GL_COLOR_ARRAY);

            //draw black rectangles
            glEnable(GL_POLYGON_OFFSET_FILL);
            glPolygonOffset(1.0, 1.0);
            glColor3f (0.0, 0.0, 0.0);
            glDrawArrays(GL_TRIANGLES, 0, Vertices.size()/9*3);
            glDisable(GL_POLYGON_OFFSET_FILL);

            glEnableClientState(GL_COLOR_ARRAY);
            //glColorPointer(3, GL_FLOAT, 0,  &Colors.front()    );

            //draw the lines
            glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

            glDrawArrays(GL_TRIANGLES, 0, Vertices.size()/9*3);

            glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

            break;
        case DRAW_TRIANGLES:

            glEnableClientState(GL_NORMAL_ARRAY);

            glNormalPointer(GL_FLOAT,0,&Normals.front());

            glDrawArrays(GL_TRIANGLES, 0, Vertices.size()/9*3);
            //glDrawElements(GL_TRIANGLES, Indices.size()/3*3, GL_UNSIGNED_INT, &Indices.front());

            glDisableClientState(GL_NORMAL_ARRAY);

            break;
        case DRAW_LINES:
            glDrawElements(GL_LINES, Indices.size()/2*2, GL_UNSIGNED_INT, &Indices.front());
            //glDrawArrays(GL_LINES, 0, Vertices.size()/6*2);
            break;
        case DRAW_POINTS:
        default:
            glDrawElements(GL_POINTS, Indices.size(), GL_UNSIGNED_INT, &Indices.front());
            //glDrawArrays(GL_POINTS, 0, Vertices.size()/3);
            break;
    }

    glDisableClientState(GL_VERTEX_ARRAY);  // disable vertex arrays
	glDisableClientState(GL_COLOR_ARRAY);

    /* finally, swap the back and front buffers */
    SDL_GL_SwapBuffers();
}


void genv::GLcanvas::setup_opengl()
{
    float aspect = (float)Width / (float)Height;

    /* Make the viewport cover the whole window */
    glViewport(0, 0, Width , Height);

    /* Set the camera projection matrix:
     * field of view: 90 degrees
     * near clipping plane at 0.1
     * far clipping plane at 100.0
     */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();

    fovy=FOVY;
    fovx=atan(   tan(fovy/180.0*M_PI)*Width/Height  )*180.0/M_PI;

    gluPerspective(fovy, aspect, NEAR_CLIPPING_DISTANCE, FAR_CLIPPING_DISTANCE);

    //calculate the distance of 1 pixel in X and Y directions
    DistancePerPixel=tan(fovy/180*M_PI/2)*2.0/Height;

    /* We're done with the camera, now matrix operations
     * will affect the modelview matrix
     * */
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glGetFloatv(GL_MODELVIEW_MATRIX,InvProjMat);


    //glShadeModel(GL_FLAT);
    glShadeModel(GL_SMOOTH);							// Enable Smooth Shading
	//glClearColor(0.5f, 0.5f, 0.5f, 0.5f);				// Black Background
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);				// Black Background

	glClearDepth(1.0f);									// Depth Buffer Setup
	glEnable(GL_DEPTH_TEST);							// Enables Depth Testing
	//glDepthFunc(GL_LESS);								// The Type Of Depth Testing To Do

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);	// Really Nice Perspective Calculations

    glFrontFace(GL_CCW);
    glCullFace(GL_BACK);

    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );



    GLfloat LightPosition[]= { 2.0f, 0.0f, 2.0f, 0.0f }; //utolso 0: mozog de nem forog; 1: nem mozduk
    glLightfv(GL_LIGHT1, GL_POSITION,LightPosition);            // Position The Light


    GLfloat LightAmbient[]= { 0.2f, 0.2f, 0.2f, 1.0f };                 // Ambient Light Values
    GLfloat LightDiffuse[]= { 1.0f, 1.0f, 1.0f, 0.0f };              // Diffuse Light Values


    glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient);             // Setup The Ambient Light
    glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse);             // Setup The Diffuse Light

    glEnable(GL_LIGHT1);                            // Enable Light One


    glEnable(GL_COLOR_MATERIAL);
    glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
}

